/* ReSizeable RAMDisk - EMS disk I/O
 * Copyright (C) 1992-1996, 2005 Marko Kohtala
 * Released under GNU GPL, read the file 'COPYING' for more information
 */

#include "srdisk.h"
#include <dos.h>
#include <stdio.h>
#include <assert.h>
#include "max.h"

/*
** EMS errors
*/

static void EMS_error(byte err)
{
  char *errstr = "Unknown error";
  int e;
  static struct {
    byte err;
    char *str;
  } errs[] = {
    { 0x80, "Driver software failure" },
    { 0x81, "Hardware failure" },
    { 0x83, "Unknown handle" },
    { 0x84, "Unknown function" },
    { 0x85, "No handles available" },
    { 0x87, "Insufficient total pages" },
    { 0x88, "Insufficient available pages" },
    { 0x89, "Zero pages requested for LIM 3.2 functions" }
  };

  for (e = 0; e < sizeof errs / sizeof errs[0]; e++)
    if (errs[e].err == err) {
      errstr = errs[e].str;
      break;
    }

  force_banner();
  printf("EMS error %02X: %s\n", err, errstr);
}

/*
**  Memory allocation support
*/

/* Local utilities */

#define GET_HANDLE(allocs) (*(word far *)((byte far *)allocs + EMS_handle))

static void far *old_allocs;
static word old_handle;
static dword old_size;

extern byte free_ems_handle(word handle);

#pragma argsused
static int EMSf_free_handle(void far *allocs, word handle)
{
  byte err = free_ems_handle(handle);
  if (!err) {
    return 1;
  }
  EMS_error(err);	/* EMS_error(_AH) gets miscompiled by bcc 3.1 */
  if (err != 0x83)        /* If unknown handle, it is freed... */
    return 0;
 ok:
  return 1;
}

/* External interface */

int EMSf_has_memory(void far *alloc)
{
  return GET_HANDLE(alloc) != (word)-1;
}

extern byte alloc_ems_handle(void far * allocs, word pages);

dword EMSf_alloc(void far *allocs, dword size)
{
  word pages;
  byte err;

  if (size > 0xFFFF0L) /* Can not allocate over 0xFFFFF K */
    size = 0xFFFF0L;

  pages = (size >> 4) + ((size & 0xF) ? 1 : 0);

  err = alloc_ems_handle(allocs, pages);
  if (!err)
    return pages << 4;
 alloc_fail:
  EMS_error(err);
//  asm { xchg al,ah; push ax; call EMS_error; pop ax; }
  pages = 0;
 alloc_ok:;
  return pages << 4;
}

extern word realloc_ems_handle(void far * allocs, word * pages);

dword EMSf_realloc(void far *allocs, dword currsize, dword size)
{
  word pages;
  word rc;
  byte err;

  if (size > 0xFFFF0L) /* Can not allocate over 0xFFFFF K */
    size = 0xFFFF0L;

  pages = (size >> 4) + ((size & 0xF) ? 1 : 0);

  rc = realloc_ems_handle(allocs, &pages);
  err = (byte)rc;
  if (!err)
    return pages << 4;
  if ( (rc >> 8) == 0x84)
    return currsize;
 error:
  EMS_error(err);
//  asm { xchg al,ah; push ax; call EMS_error; pop ax; }
 ok:
  return pages << 4;
 not_implemented:
  return currsize;
}

dword EMSf_free(void far *allocs, dword currsize)
{
  if (EMSf_free_handle(allocs, GET_HANDLE(allocs))) {
    GET_HANDLE(allocs) = (word)-1;
    return 0;
  }
  return currsize;
}

/******************* realloc implementation functions ****************/

void EMSf_save_buffer(void far *alloc, dword size)
{
  old_allocs = alloc;
  old_handle = GET_HANDLE(alloc);
  GET_HANDLE(alloc) = (word)-1;
  old_size = size;
}

extern byte copy_ems(void far * pmb);

int EMSf_copy_to_new(void far *alloc, dword size)
{
  struct __attribute__((__packed__)) EMS_move_s {
    dword length;
    byte stype;
    word shandle;
    word soff;
    word spage;
    byte dtype;
    word dhandle;
    word doff;
    word dpage;
  } mb;
  struct EMS_move_s far * pmb = &mb;
  byte err;

  assert(alloc == old_allocs);

  mb.length = min(old_size, size) * 1024;
  mb.stype = 1;
  mb.shandle = old_handle;
  mb.soff = 0;
  mb.spage = 0;
  mb.dtype = 1;
  mb.dhandle = GET_HANDLE(alloc);
  mb.doff = 0;
  mb.dpage = 0;
  err = copy_ems(pmb);
  if (!err)
    return 1;
  EMS_error(err);
//  asm { xchg al,ah; push ax; call EMS_error; pop ax; }
  return 0;
 move_ok:
  return 1;
}

int EMSf_free_old(void)
{
  return EMSf_free_handle(old_allocs, old_handle);
}

dword EMSf_restore_buffer(void far *alloc)
{
  if (GET_HANDLE(alloc) != (word)-1)
    EMSf_free(alloc, 1);
  GET_HANDLE(old_allocs) = old_handle;
  return old_size;
}

/* Memory size determination */

extern dword ems_available(void);

#pragma argsused
dword EMSf_mem_avail(void far *alloc)
{
  return ems_available();
}
