;page 255,132

; Copyright (C) 1983 Microsoft Corp.
; Copyright (C) IBM and Microsoft Corporation.
; Modifications copyright 2018 John Elliott
;           and copyright 2022 S. V. Nickolas.
;
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the Software), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
;
; The above copyright notice and this permission notice shall be included in
; all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED AS IS, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
; FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
; IN THE SOFTWARE.
;
; MS-DOS is a Registered Trademark of Microsoft Corp.

; Routines to perform debugger commands except ASSEMble and UASSEMble

          %include "debequ.mac"
          %include "dossym.mac"

code      segment   public word 'code'
code      ends

const     segment   public word
          extrn     notfnd:byte,noroom:byte,drvlet:byte,nospace:byte,nambad:byte
          extrn     toobig:byte,errmes:byte
          extrn     exebad:byte,hexerr:byte,exewrt:byte,hexwrt:byte
          extrn     execemes:byte,wrtmes1:byte,wrtmes2:byte,accmes:byte

          extrn     flagtab:word,exec_block:byte,com_line:dword,com_fcb1:dword
          extrn     com_fcb2:dword,com_sssp:dword,com_csip:dword,retsave:word
          extrn     newexec:byte
          extrn     regtab:byte,totreg:byte,noregl:byte
          extrn     user_proc_pdb:word,stack:byte,rstack:word,axsave:word
          extrn     bxsave:word,dssave:word,essave:word,cssave:word,ipsave:word
          extrn     sssave:word,cxsave:word,spsave:word,_fsave:word
          extrn     sreg:byte,segtab:word,regdif:word,rdflg:byte
	extrn	linebuf:byte
const     ends

data      segment   public word
          extrn     defdump:byte,transadd:dword,index:word,buffer:byte
          extrn     asmadd:byte,disadd:byte,nseg:word,bptab:byte
          extrn     brkcnt:word,tcount:word,xnxcmd:byte,xnxopt:byte
          extrn     aword:byte,extptr:word,handle:word,parserr:byte
	extrn	zpcount:word
	extrn	int1save:word,int1saveseg:word
	extrn	int3save:word,int3saveseg:word
	extrn	bytebuf:byte
	extrn	gethexhigh:word
	extrn	diskpacket:byte, dp_sector_low:word, dp_sector_high:word
	extrn	dp_amount:word, dp_offset:word, dp_segment:word
	extrn	usepacket:byte, usefat32packet:byte, disknumber:byte
	extrn	switchar:byte
data      ends

dg        group     code,const,data

code      segment   public word 'code'
assume    cs:dg,ds:dg,es:dg,ss:dg

          public    defio,skip_file,debug_found
          public    reg,compare,go,input,load
          public    kk,kk_si,_name,output,trace,ztrace,dwrite
	public	saveints, restoreints
	public	closefile

	extrn	jump_error:near
	extrn	programloadname:byte
          extrn     gethex:near,geteol:near
          extrn     crlf:near,blank:near,_out:near
          extrn     outsi:near,outdi:near,inbuf:near,scanb:near,scanp:near
          extrn     rprbuf:near,hex:near,out16:near,digit:near
          extrn     command:near,disasln:near,set_terminate_vector:near
          extrn     restart:near,dabort:near,terminate:near,drverr:near
          extrn     find_debug:near,nmiint:near,nmiintend:near
          extrn     hexchk:near,gethex1:near,print:near,dsrange:near
          extrn     address:near,hexin:near,perror:near

debcom2:
dispreg:
          mov       si,offset regtab wrt dg
          mov       bx,offset axsave wrt dg
          mov       byte [totreg],13
          mov       ch,0
          mov       cl,[noregl]
repdisp:
          sub       [totreg],cl
          call      dispregline
          call      crlf
          mov       ch,0
          mov       cl,[noregl]
          cmp       cl,[totreg]
          jl        repdisp
          mov       cl,[totreg]
          call      dispregline
          call      blank
          call      dispflags
          call      crlf
          mov       ax,[ipsave]
          mov       word ptr [disadd],ax
          push      ax
          mov       ax,[cssave]
          mov       word ptr [disadd+2],ax
          push      ax
          mov       word [nseg],-1
          call      disasln
          pop       word [disadd+2]
          pop       word [disadd]
          mov       ax,[nseg]
          cmp       al,-1
          jz        crlfj
          cmp       ah,-1
          jz        noover
          xchg      al,ah
noover:
          cbw
          mov       bx,ax
          shl       bx,1
          mov       ax,word ptr [bx+sreg]
          call      _out
          xchg      al,ah
          call      _out
          mov       al,":"
          call      _out
          mov       dx,[index]
          call      out16
          mov       al,"="
          call      _out
          mov       bx,[bx+segtab]
          push      ds
          mov       ds,[bx]
          mov       bx,dx
          mov       dx,[bx]
          pop       ds
          test      byte ptr [aword],-1
          jz        out8
          call      out16
crlfj:
          jmp       crlf
out8:
          mov       al,dl
          call      hex
          jmp       crlf

dispregj: jmp       dispreg

; Perform register dump if no parameters or set register if a
; register designation is a parameter.

reg:
          call      scanp
          jz        dispregj
          mov       dl,[si]
          inc       si
          mov       dh,[si]
          cmp       dh,13
          jz        flag
          inc       si
          call      geteol
          cmp       dh," "
          jz        flag
          mov       di,offset regtab wrt dg
          xchg      ax,dx
          push      cs
          pop       es
          mov       cx,regtablen
          repnz     scasw
          jnz       badreg
          or        cx,cx
          jnz       notpc
          dec       di
          dec       di
          mov       ax,[cs:di-2]
notpc:
          call      _out
          mov       al,ah
          call      _out
          call      blank
          push      ds
          pop       es
          lea       bx,[di+regdif-2]
          mov       dx,[bx]
          call      out16
          call      crlf
          mov       al,":"
          call      _out
          call      inbuf
          call      scanb
          jz        ret4
          mov       cx,4
          call      gethex1
          call      geteol
          mov       [bx],dx
ret4:     ret
badreg:
          mov       ax,5200h+'B'            ; BR ERROR
          jmp       err
          nop				; NASM identicalise
flag:
          cmp       dl,"F"
          jnz       badreg
          call      dispflags
          mov       al,"-"
          call      _out
          call      inbuf
          call      scanb
          xor       bx,bx
          mov       dx,[_fsave]
getflg:
          lodsw
          cmp       al,13
          jz        savchg
          cmp       ah,13
          jz        flgerr
          mov       di,offset flagtab wrt dg
          mov       cx,32
          push      cs
          pop       es
          repne     scasw
          jnz       flgerr
          mov       ch,cl
          and       cl,0fh
          mov       ax,1
          rol       ax,cl
          test      ax,bx
          jnz       repflg
          or        bx,ax
          or        dx,ax
          test      ch,16
          jnz       nexflg
          xor       dx,ax
nexflg:
          call      scanp
          jmp       short getflg
dispregline:
          cs lodsw
          call      _out
          mov       al,ah
          call      _out
          mov       al,'='
          call      _out
          mov       dx,[bx]
          inc       bx
          inc       bx
          call      out16
          call      blank
          call      blank
          loop      dispregline
          ret
repflg:
          mov       ax,4600h+'D'            ; DF ERROR
ferr:
          call      savchg
err:
          call      _out
          mov       al,ah
          call      _out
          mov       si,offset errmes wrt dg
          jmp       print
savchg:
          mov       [_fsave],dx
          ret
flgerr:
          mov       ax,4600h+'B'            ; BF ERROR
          jmp short ferr
dispflags:
          mov       si,offset flagtab wrt dg
          mov       cx,16
          mov       dx,[_fsave]
dflags:
          cs lodsw
          shl       dx,1
          jc        flagset
          mov       ax,[cs:si+30]
flagset:
          or        ax,ax
          jz        nextflg
          call      _out
          mov       al,ah
          call      _out
          call      blank
nextflg:
          loop      dflags
          ret

; Input from the specified port and display result

input:
          mov       cx,4                    ; Port may have 4 digits
          call      gethex                  ; Get port number in DX
          call      geteol
          in        al,dx                   ; Variable port input
          call      hex                     ; And display
          jmp       crlf

; Output a value to specified port.

output:
          mov       cx,4                    ; Port may have 4 digits
          call      gethex                  ; Get port number
          push      dx                      ; Save while we get data
          mov       cx,2                    ; Byte output only
          call      gethex                  ; Get data to output
          call      geteol
          xchg      ax,dx                   ; Output data in AL
          pop       dx                      ; Port in DX
          out       dx,al                   ; Variable port output
ret5:     ret
compare:
          call      dsrange
          push      cx
          push      ax
          push      dx
          call      address                 ; Same segment
          call      geteol
          pop       si
          mov       di,dx
          mov       es,ax
          pop       ds
          pop       cx                      ; Length
          dec       cx
          call      comp                    ; Do one less than total
          inc       cx                      ; CX=1 (do last one)
comp:
          repe      cmpsb
          jz        ret5
          dec       si                      ; Compare error. Print address, value; value, address.
          call      outsi
          call      blank
          call      blank
          lodsb
          call      hex
          call      blank
          call      blank
          dec       di
          mov       al,[es:di]
          call      hex
          call      blank
          call      blank
          call      outdi
          inc       di
          call      crlf
          xor       al,al		; ZR in case repe cmpsb runs with cx = 0
          jmp short comp

ztrace:
; just like trace except skips OVER next INT or CALL.
          call      setadd                  ; get potential starting point
          call      scanp
          call      hexin
          mov       dx,1
          jc        zpstocnt
          mov       cx,4
          call      gethex
zpstocnt:
          mov       [zpcount],dx
          call      geteol                  ; check for end of line
zpstep:
          mov       word [tcount],1              ; only a single go at it
          mov       es,[cssave]             ; point to instruction to execute
          mov       di,[ipsave]             ; include offset in segment
          xor       dx,dx                   ; where to place breakpoint
	xor bx, bx			; no string repeat prefix
.loop:
          mov       al,[es:di]              ; get the opcode
		; lock prefix is not valid for any of our handled instructions.
	cmp al, 26h			; seg prefixes ?
	je .next
	cmp al, 2Eh
	je .next
	cmp al, 36h
	je .next
	cmp al, 3Eh
	je .next			; yes, just loop -->
	cmp al, 0F2h
	je .rep
	cmp al, 0F3h
	jne .noprefix
.rep:
	mov bl, 1			; mark rep/repne prefix as present
.next:
	inc di				; -> next opcode byte
	jmp .loop

.noprefix:
          cmp       al,11101000b            ; direct intra call (E8h)
          jz        ztrace3                 ; yes, 3 bytes
          cmp       al,10011010b            ; direct inter call (9Ah)
          jz        ztrace5                 ; yes, 5 bytes
          cmp       al,11111111b            ; indirect? (FFh)
          jz        ztracemodrm             ; yes, go figure length
	cmp al, 0CEh			; into ?
	je ztrace1
          cmp       al,11001100b            ; short interrupt? (CCh)
          jz        ztrace1                 ; yes, 1 byte
          cmp       al,11001101b            ; long interrupt? (CDh)
          jz        ztrace2                 ; yes, 2 bytes
          cmp       al,11100010b            ; loop (E2h)
          jz        ztrace2                 ; 2 byter
          cmp       al,11100001b            ; loopz/loope (E1h)
          jz        ztrace2                 ; 2 byter
          cmp       al,11100000b            ; loopnz/loopne (E0h)
          jz        ztrace2                 ; 2 byter
	test bl, bl			; repeat prefix ?
	jz step_j			; no, do not handle single string inst -->
          and       al,11111110b            ; ignore w bit
          cmp       al,10100100b            ; MOVS (A4h)
	je ztrace1			; one byte, es:di -> past prefixes already
          cmp       al,10100110b            ; CMPS (A6h)
	je ztrace1
          cmp       al,10101110b            ; SCAS (AEh)
	je ztrace1
          cmp       al,10101100b            ; LODS (ACh)
	je ztrace1
          cmp       al,10101010b            ; STOS (AAh)
	je ztrace1
step_j:
          jmp       step                    ; bogus, do single step

ztracemodrm:
          mov       al,[es:di+1]            ; get next byte
	mov ah, al
	and ax, 00_111_000_11_000_111b	; get /r, and mod and r/m
	cmp ah, 3 << 3			; call far FFh /3 ?
	je .far
	cmp ah, 2 << 3			; call near FFh /2 ?
	jne step_j			; no -->
.near:
	cmp al, 11_000_000b		; mod 3 (register) ?
	jae ztrace2			; yes, just 2 to skip -->
.far:
	cmp al, 11_000_000b		; mod 3 (register) ?
	jae step_j			; for far this is invalid, step it -->
	cmp al, 10_000_000b		; mod 2 (disp16) ?
	jae ztrace4
	cmp al, 01_000_000b		; mod 1 (disp8) ?
	jae ztrace3
	cmp al, 00_000_000b + 6		; mod 0 (no disp) with BP encoding ?
	je ztrace4			; yes, special case 16-bit offset -->
	jmp ztrace2			; mod 0 (no disp), not BP encoding -->

ztrace5:  inc       dx
ztrace4:  inc       dx
ztrace3:  inc       dx
ztrace2:  inc       dx
ztrace1:  inc       dx
          add       di,dx                   ; offset to breakpoint instruction
          mov       word ptr [bptab],di     ; save offset
          mov       word ptr [bptab+2],es   ; save segment
          mov       al,[es:di]              ; get next opcode byte
          mov       byte ptr [bptab+4],al   ; save it
          mov       byte ptr [es:di],0cch   ; break point it
          mov       word [brkcnt],1              ; only this breakpoint
          jmp       dexit                   ; start the operation!

; Trace 1 instruction or the number of instruction specified
; by the parameter using 8086 trace mode. Registers are all
; set according to values in save area

trace:
          call      setadd
          call      scanp
          call      hexin
          mov       dx,1
          jc        stocnt
          mov       cx,4
          call      gethex
stocnt:
          mov       [tcount],dx
          call      geteol
step_zp_reset:
          mov       word [zpcount], 1
step:
          mov       es,[cssave]             ; point to instruction to execute
          mov       di,[ipsave]             ; include offset in segment
	mov bx, 3			; default for int3
	mov dx, 1			; default for int3 or into
          mov       al,[es:di]              ; get the opcode
	cmp al, 0CDh
	je stepint
	cmp al, 0CCh
	je stepint_3
	cmp al, 0CEh
	jne steptrace
stepint_o:
	test byte ptr [_fsave + 1], 8	; OV ?
	jz steptrace			; no -->
	inc bx				; = 4
	jmp stepint_common
          nop				; NASM identicalise

stepint:
	mov bl, byte ptr [es:di + 1]	; get interrupt number
	inc dx				; = 2

		; Note that this doesn't work in a sensible way.
		;  In Microsoft's Debug it would trace into the
		;  debugger's own interrupt 3 handler, unless no
		;  steptrace/go/zptrace command had run yet.
		; In that exceptional case the T command would
		;  trace into the prior interrupt 3 handler. This
		;  is true always for our current source because
		;  we restore the original interrupt 3 handler
		;  when we are re-entered after running.
		; Both of those are not desirable.
; stepint_3:
		; INP:	es:di = saved cs:ip
		;	dx = length of instruction
		;	bx = interrupt number
stepint_common:
	mov si, word ptr [spsave]
	mov ds, word ptr [sssave]
	add di, dx			; = new ip after interrupt
	dec si
	dec si
	mov ax, word ptr [cs:_fsave]
	mov word ptr [si], ax		; store flags
	dec si
	dec si
	mov word ptr [si], es		; store cs
	dec si
	dec si
	mov word ptr [si], di		; store new ip
	xor ax, ax
	mov ds, ax			; => IVT
	add bx, bx
	add bx, bx			; -> IVT entry of our interrupt
	 mov ax, word ptr [bx]
	 mov word ptr [cs:ipsave], ax	; save new ip
	 mov ax, word ptr [bx + 2]
	push cs
	pop ds				; reset our ds
	 mov word ptr [cssave], ax	; save new cs
	mov word ptr [spsave], si	; save new sp
	and byte ptr [_fsave + 1], ~ 3
					; clear IF and TF
j_stepint_continue:
	push cs
	pop es				; reset our es
	jmp stepint_continue

		; New: Just skip past the int3 instruction
		;  if we encounter it in the T command.
stepint_3:
	inc word ptr [ipsave]
	jmp j_stepint_continue


steptrace:
          mov       word [brkcnt],0
          or        byte ptr [_fsave+1],1
dexit:
          mov       bx,[user_proc_pdb]
set_current_pdb equ Set_Current_PDB	; NASM port equate
          mov       ah,set_current_pdb
          int       21h
	call setint
restoredint:
	jmp restoredint_actual

setint:
	pop ax		; discard near return address
          push      ds
	call saveints

          mov       word ptr [12],offset breakfix wrt dg ; Set vector 3--breakpoint instruction
          mov       word ptr [14],cs
          mov       word ptr [4],offset reenter wrt dg   ; Set vector 1--Single step
          mov       word ptr [6],cs

          %if        setcntc
          mov       word ptr [8ch],offset contc wrt dg   ; Set vector 23H (CTRL-C)
          mov       word ptr [8eh],cs
          %endif

          pop       ds
          mov       sp,offset stack wrt dg
          pop       ax
          pop       bx
          pop       cx
          pop       dx
          pop       bp
          pop       bp
          pop       si
          pop       di
          pop       es
          pop       es
          pop       ss
          mov       sp,[spsave]
          push      word [_fsave]
          push      word [cssave]
          push      word [ipsave]
          mov       ds,[dssave]
          iret
step1:
          call      crlf
          call      dispreg
          jmp       step_zp_reset

zpstep1:
          call      crlf
          call      dispreg
          jmp       zpstep

; Re-entry point from CTRL-C. Top of stack has address in 86-DOS for
; continuing, so we must pop that off.

contc:
          add       sp,6
          jmp       short reenterreal

; Re-entry point from breakpoint. Need to decrement instruction
; pointer so it points to location where breakpoint actually
; occured.

breakfix:
          push      bp
          mov       bp,sp
          dec       word ptr [bp + oldip]
          nop				; NASM identicalise
          pop       bp
          jmp       reenterreal
          nop				; NASM identicalise

; Re-entry point from trace mode or interrupt during
; execution. All registers are saved so they can be
; displayed or modified.

interrupt_frame struc
oldbp   dw  ?
oldip   dw  ?
oldcs   dw  ?
oldf    dw  ?
olderip dw  ?
oldercs dw  ?
olderf  dw  ?
interrupt_frame ends

reenter:
          push      bp
          mov       bp,sp                   ; get a frame to address from
          push      ax
          mov       ax,cs
          cmp       ax,[bp + oldcs]           ; Did we interrupt ourselves?
          jnz       goreenter               ; no, go reenter
          mov       ax,[bp + oldip]
          cmp       ax,offset nmiint wrt dg     ; interrupt below NMI interrupt?
          jb        goreenter               ; yes, go reenter
          cmp       word [bp + oldip],offset nmiintend wrt dg
          jae       goreenter               ; interrupt above NMI interrupt?
          pop       ax                      ; restore state
          pop       bp
          sub       sp,6                    ; switch TRACE and NMI stack frames
          push      bp
          mov       bp,sp                   ; set up frame
          push      ax                      ; get temp variable
          mov       ax,[bp + olderip]         ; get NMI Vector
          mov       word [bp + oldip],ax           ; stuff in new NMI vector
          mov       ax,[bp + oldercs]         ; get NMI Vector
          mov       word [bp + oldcs],ax           ; stuff in new NMI vector
          mov       ax,[bp + olderf]          ; get NMI Vector
          and       ah,0feh                 ; turn off Trace if present
          mov       word [bp + oldf],ax            ; stuff in new NMI vector
          mov       word [bp + olderf],ax
          mov       word [bp + olderip],offset reenter wrt dg  ; offset of routine
          mov       word [bp + oldercs],cs         ; and cs
          pop       ax
          pop       bp
          iret                              ; go try again
goreenter:
          pop       ax
          pop       bp
reenterreal:
          mov       [cs:spsave+segdif],sp
          mov       [cs:sssave+segdif],ss
          mov       [cs:_fsave],cs
          mov       ss,[cs:_fsave]
          mov       sp,offset rstack wrt dg
          push      es
          push      ds
          push      di
          push      si
          push      bp
          dec       sp
          dec       sp
          push      dx
          push      cx
          push      bx
          push      ax
          push      ss
          pop       ds
          mov       ss,[sssave]
          mov       sp,[spsave]
          pop       word [ipsave]
          pop       word [cssave]
          pop       ax
          and       ah,0feh                 ; turn off trace mode bit
          mov       [_fsave],ax
          mov       [spsave],sp
          push      ds
          pop       es
          push      ds
          pop       ss
          mov       sp,offset stack wrt dg
          push      ds
	call restoreints

          %if        setcntc
          mov       word ptr [8ch],offset dabort wrt dg  ; Set Ctrl-C vector
          mov       word ptr [8eh],cs
          %endif

          pop       ds
	jmp restoredint

restoredint_actual:
          sti
          cld
get_current_pdb equ Get_Current_PDB	; NASM port equate
          mov       ah,get_current_pdb
          int       21h
          mov       [user_proc_pdb],bx
          mov       bx,ds
          mov       ah,set_current_pdb
          int       21h
stepint_continue:
          dec       word [tcount]
          jz        checkdisp
          jmp       step1
checkdisp:
          mov       si,offset bptab wrt dg
          mov       cx,[brkcnt]
          jcxz      shoreg
          push      es
clearbp:
          les       di,[si]
          add       si,4
          movsb
          loop      clearbp
          pop       es
shoreg:
          dec       word [zpcount]
          jz        notzpcount
          jmp       zpstep1
notzpcount:
          call      crlf
          call      dispreg
          jmp       command

setadd:
          mov       bp,[cssave]
          call      scanp
          cmp       byte ptr [si],'='
          jnz       ret$5
          inc       si
          call      address
          mov       [cssave],ax
          mov       [ipsave],dx
ret$5:    ret

; Jump to program, setting up registers according to the
; save area. up to 10 breakpoint addresses may be specified.

go:
          call      setadd
          xor       bx,bx
          mov       di,offset bptab wrt dg
go1:
          call      scanp
          jz        dexec
          cmp bx, bpmax
          jae bperror
          mov       bp,[cssave]
          call      address
          mov       [di],dx                 ; Save offset
          mov       [di+2],ax               ; Save segment
          add       di,5                    ; Leave a little room
          inc       bx
          jmp go1
bperror:
          mov       ax,5000h+'B'            ; bp error
          jmp       err
dexec:
          mov       [brkcnt],bx
          mov       cx,bx
          jcxz      nobp
          mov       di,offset bptab wrt dg
          push      ds
setbp:
          lds       si,[es:di]
          add       di,4
          movsb
          mov       byte ptr [si-1],0cch
          loop      setbp
          pop       ds
nobp:
          mov       word [tcount],1
          mov       word [zpcount], 1
          jmp       dexit

saveints:
          xor       ax,ax
          mov       ds,ax
	cli
	mov ax, word ptr [4]
	mov word ptr [cs:int1save], ax
	mov ax, word ptr [6]
	mov word ptr [cs:int1saveseg], ax
	mov ax, word ptr [12]
	mov word ptr [cs:int3save], ax
	mov ax, word ptr [14]
	mov word ptr [cs:int3saveseg], ax
	ret

restoreints:
          xor       ax,ax
          mov       ds,ax
	cli
	mov ax, word ptr [cs:int1save]
	mov word ptr [4], ax
	mov ax, word ptr [cs:int1saveseg]
	mov word ptr [6], ax
	mov ax, word ptr [cs:int3save]
	mov word ptr [12], ax
	mov ax, word ptr [cs:int3saveseg]
	mov word ptr [14], ax
	ret

skip_file:
	MOV	AH, Char_Oper
	INT	21H
	MOV	[CS:switchar], DL	; GET THE CURRENT SWITCH CHARACTER
find_delim:
          lodsb
          call      delim1
          jz        gotdelim
          call      delim2
          jnz       find_delim
gotdelim:
          dec       si
ret23:    ret


;  OPENS A XENIX PATHNAME SPECIFIED IN THE UNFORMATTED PARAMETERS
;  VARIABLE [XNXCMD] SPECIFIES WHICH COMMAND TO OPEN IT WITH
;
;  VARIABLE [HANDLE] CONTAINS THE HANDLE
;  VARIABLE [EXTPTR] POINTS TO THE FILES EXTENSION

delete_a_file:
unlink equ Unlink	; NASM port equate
          mov       byte ptr [xnxcmd],unlink
          jmp       short oc_file

parse_a_file:
          mov       byte ptr [xnxcmd],0
          jmp       short oc_file

exec_a_file:
exec equ Exec	; NASM port equate
          mov       byte ptr [xnxcmd],exec
          mov       byte ptr [xnxopt],1
          jmp       short oc_file

open_a_file:
open equ Open	; NASM port equate
          mov       byte ptr [xnxcmd],open
          mov       byte ptr [xnxopt],2     ; Try read write
          call      oc_file
          jnc       ret23
          mov       byte ptr [xnxcmd],open
          mov       byte ptr [xnxopt],0     ; Try read only
          jmp       short oc_file

create_a_file:
creat equ Creat	; NASM port equate
          mov       byte ptr [xnxcmd],creat

oc_file:
          push      ds
          push      es
          push      ax
          push      bx
          push      cx
          push      dx
          push      si
          xor       ax,ax
          mov       [extptr],ax             ; INITIALIZE POINTER TO EXTENSIONS
	MOV	AH, Char_Oper
	INT	21H
	MOV	[CS:switchar], DL	; GET THE CURRENT SWITCH CHARACTER
          mov       si, offset programloadname wrt dg

open1:    call      getchrup
          call      delim2                  ; END OF LINE?
          jz        open4
          call      delim1                  ; SKIP LEADING DELIMITERS
          jz        open1

          mov       dx,si                   ; SAVE POINTER TO BEGINNING
          dec       dx
open2:    cmp       al,'.'                  ; LAST CHAR A "."?
          jnz       open3
          mov       [extptr],si             ; SAVE POINTER TO THE EXTENSION
open3:    call      getchrup
          call      delim1                  ; LOOK FOR END OF PATHNAME
          jz        open4
          call      delim2
          jnz       open2

open4:    dec       si                      ; POINT BACK TO LAST CHAR
          push      word [si]                    ; SAVE TERMINATION CHAR
          mov       byte ptr [si],0         ; NULL TERMINATE THE STRING

          mov       al,[xnxopt]
          mov       ah,[xnxcmd]             ; OPEN OR CREATE FILE
          or        ah,ah
          jz        opnret
          mov       bx,offset exec_block wrt dg
          xor       cx,cx
	cmp ah, open
	je open_with_handle
	cmp ah, creat
	je open_with_handle

open_without_handle:
	int 21h
	jmp opnret
          nop				; NASM identicalise

open_with_handle:
	int 21h
	jc opnret
          mov       [cs:handle],ax          ; SAVE ERROR CODE OR HANDLE

opnret:   pop       word [si]

          pop       si
          pop       dx
          pop       cx
          pop       bx
          pop       ax
          pop       es
          pop       ds
          ret

getchrup:
          lodsb
          cmp       al,"a"
          jb        gcur
          cmp       al,"z"
          ja        gcur
          sub       al,32
          mov       [si-1],al
gcur:     ret

delim0:   cmp       al,'['
          jz        limret
delim1:   cmp       al,' '                  ; SKIP THESE GUYS
          jz        limret
          cmp       al,';'
          jz        limret
          cmp       al,'='
          jz        limret
          cmp       al,9
          jz        limret
          cmp       al,','
          retn

delim2:
	CMP AL, [CS:switchar]	; STOP ON THESE GUYS
          je        limret
          cmp       al,13
limret:   ret

copy_to_programloadname:
	push si
	mov di, offset programloadname wrt dg
	mov cx, 128 / 2
	rep movsw
	pop si
	ret

kk:
	sub si, offset bytebuf wrt dg
	add si, offset linebuf wrt dg	; get original string (not capitalised)
kk_si:
	mov bx, si
	call copy_to_programloadname
; parschk:
          mov       di,fcb
parse_file_descriptor equ Parse_File_Descriptor	; NASM port equate
          mov       ax,(parse_file_descriptor << 8) | 01h
          int       21h
          mov       byte ptr [parserr],al
					; writes to parserr for kk
		; This variable indicates whether the
		;  program load name is valid. For kk
		;  that means it must be written here,
		;  not after processing the 5Ch FCB.
          call      skip_file           ; Make sure si points to delimiter
; prepname:
	mov di, si
	neg bx
	add bx, di			; offset of tail - offset start
	call copycomtail

	mov cx, 100h
	sub cx, di
	mov al, 0
	rep stosb			; clear trailer
	lea si, [bx + programloadname wrt dg]

kk_nn_common:
          mov       di,fcb
          mov       ax,(parse_file_descriptor << 8) | 01h
          int       21h
          cmp al, 1
          jne .no_1_first_fcb
          dec ax
.no_1_first_fcb:
          mov       byte ptr [axsave],al    ; Indicate analysis of first parm

          call      skip_file
          mov       di,6ch
          mov       ax,(parse_file_descriptor << 8) | 01h
          int       21h
          cmp al, 1
          jne .no_1_second_fcb
          dec ax
.no_1_second_fcb:
          mov       byte ptr [axsave+1],al  ; Indicate analysis of second parm
          ret

copycomtail:
          mov       di,81h
comtail:
          lodsb
          stosb
          cmp       al,13
          jnz       comtail
	mov ax, di
          sub       al,82h
          mov       byte ptr [es:80h],al
	ret

_name:
	call copy_to_programloadname
          mov       es,[dssave]
          push      si
	call copycomtail
          pop       si
	call kk_nn_common
          mov       al,byte ptr [axsave]
          mov       [parserr],al
          push      es
          pop       ds
          push      cs
          pop       es
          mov       si,fcb                  ; DS:SI points to user FCB
          mov       di,si                   ; ES:DI points to DEBUG FCB
          mov       cx,82
          rep       movsw
ret6:     ret

badnam:
          mov       dx,offset nambad wrt dg
          jmp       restart

ifhex:
		; parserr may hold 0, 1, or -1
          cmp       byte ptr [parserr],-1   ; Invalid drive specification?
          jz        badnam
          call      parse_a_file
          mov       bx,[extptr]
          cmp       word ptr [bx],'HE'   ; "HE"	; NASM port swapped text literals
          jnz       ret6
          cmp       byte ptr [bx+2],'X'
          ret

ifexe:
          push      bx
          mov       bx,[extptr]
          cmp       word ptr [bx],'EX'   ; "EX"	; NASM port swapped text literals
          jnz       retif
          cmp       byte ptr [bx+2],'E'
retif:    pop       bx
          ret

load:
read equ Read	; NASM port equate
          mov       byte ptr [rdflg],read
          jmp       short dskio

dwrite:
write equ Write	; NASM port equate
          mov       byte ptr [rdflg],write
dskio:
          mov       bp,[cssave]
          call      scanb
          jnz       primio
          jmp       defio
primio:   call      address
          call      scanb
          jnz       prmio
          jmp       fileio
prmio:    push      ax                      ; Save segment
	mov word ptr [dp_segment], ax
	mov word ptr [dp_offset], dx
          mov       bx,dx                   ; Put displacement in proper register
          mov       cx,2
          call      gethex                  ; Drive number may be 2 digits
          push      dx
          mov       cx,8
          call      gethex                  ; Logical record number
	push word ptr [gethexhigh]
	pop word ptr [dp_sector_high]
	mov word ptr [dp_sector_low], dx
	rol byte ptr [usepacket], 1
	jc pio_allow_high
	cmp word ptr [dp_sector_high], 0
	je pio_allow_high
	jmp jump_error

pio_allow_high:
          push      dx
          mov       cx,3
          call      gethex                  ; Number of records
          call      geteol
	mov word ptr [dp_amount], dx
          mov       cx,dx
          pop       dx                      ; Logical record number
          pop       ax                      ; Drive number
          cbw                               ; Turn off verify after write
          mov       byte ptr [drvlet],al      ; Save drive in case of error
          push      ax
          push      bx
          push      dx
          mov       dl,al
          inc       dl
	mov byte [disknumber], dl
	mov ah, Disk_Reset
	int 21h
	rol byte ptr [usefat32packet], 1
	jnc .oldgetdpb
	push cx
	mov ax, 7302h			; get FAT32 compatible (E)DPB
	mov cx, 48h			; (pick from MS-DOS v7.10 Debug)
	mov di, buffer
	int 21h
	pop cx
	pop dx
	pop bx
	jc drverrj
	jmp .commondpb
.oldgetdpb:

get_dpb equ Get_DPB	; NASM port equate
          mov       ah,get_dpb
          int       21h
          pop       dx
          pop       bx
          or        al,al		; NZ ?
          jnz       drverrj		; yes, drverr with al NZ (disk error)
.commondpb:
          pop       ax
          pop       ds                      ; Segment of transfer
          cmp       byte ptr [cs:rdflg],write
          jz        abswrt
	rol byte ptr [cs:usepacket], 1
	jnc absrd_direct
	call setuppacket
	rol byte ptr [cs:usefat32packet], 1
	jnc absrd_direct
absrd_fat32:
	xor si, si
	xchg dx, ax
	inc dx
	mov ax, 7305h
	int 21h
	pushf
	jmp endabs

absrd_direct:
          int       25h                     ; Primitive disk read
          jmp short endabs

setuppacket:
	push cs
	pop ds				; ds => debugger segment
	mov bx, offset diskpacket wrt dg	; ds:bx -> packet
	mov cx, 0FFFFh			; cx = 0FFFFh
	ret

; uso: Some things to note for the fix to support BIGFAT devices based on RBIL
;
; If the call comes back with CY+AX=0207h:
;   set CX=0FFFFh
;   point DS:BX to a struct as follows
;     +0 - DWORD starting sector (first 2 bytes are your old DX)
;     +4 - WORD  sector count (your old CX)
;     +6 - DWORD target address (your old DS:BX).

abswrt:
	push ax
	push bx
	push cx
	push dx
	xor bx, bx
	mov bl, byte [cs:disknumber]
	xor dx, dx
	mov cx, 084Ah
	mov ax, 440Dh			; lock
	int 21h
	pop dx
	pop cx
	pop bx
	pop ax

	rol byte ptr [cs:usepacket], 1
	jnc abswrt_direct
	call setuppacket
	rol byte ptr [cs:usefat32packet], 1
	jnc abswrt_direct
abswrt_fat32:
	mov si, 1
	xchg dx, ax
	inc dx
	mov ax, 7305h
	int 21h
	pushf
	jmp endabswrt
abswrt_direct:
          int       26h                     ; Primitive disk write
endabswrt:
	pushf
	push ax
	xor bx, bx
	mov bl, byte [cs:disknumber]
	mov cx, 086Ah
	mov ax, 440Dh			; unlock
	int 21h
	pop ax
	popf
endabs:
          jnc       ret0
	mov ah, Disk_Reset
	int 21h
	popf
	cld
drverrj:  jmp       drverr

ret0:
	mov ah, Disk_Reset
	int 21h
          popf
	cld
          ret

defio:
          mov       ax,[cssave]             ; Default segment
          mov       dx,100h                 ; Default file I/O offset
          call      ifhex
          jnz       exechk
          xor       dx,dx                   ; If HEX file, default OFFSET is zero
hex2binj: jmp       hex2bin

fileio:
; AX and DX have segment and offset of transfer, respectively
          call      ifhex
          jz        hex2binj
exechk:
          call      ifexe
          jnz       binfil
          cmp       byte ptr [rdflg],read
          jz        exelj
          mov       dx,offset exewrt wrt dg
          jmp       restart                 ; Can't write .EXE files

binfil:
          cmp       byte ptr [rdflg],write
          jz        binload
          cmp       word ptr [bx],4f00h+'C'    ; "CO"
          jnz       binload
          cmp       byte ptr [bx+2],'M'
          jnz       binload
exelj:
          dec       si
          cmp       dx,100h
          jnz       prer
          cmp       ax,[cssave]
          jz        oaf
prer:     jmp       perror
oaf:      call      open_a_file
          jnc       gdopen
          mov       ax,exec_file_not_found
          jmp       execerr

gdopen:   xor       dx,dx
          xor       cx,cx
          mov       bx,[handle]
          mov       al,2
lseek equ LSeek	; NASM port equate
          mov       ah,lseek
          int       21h
          call      ifexe                   ; SUBTRACT 512 BYTES FOR EXE
          jnz       bin2                    ; FILE LENGTH BECAUSE OF
          sub       ax,512                  ; THE HEADER
bin2:     mov       [bxsave],dx             ; SET UP FILE SIZE IN DX:AX
          mov       [cxsave],ax
	call closefile
          jmp       exeload

no_mem_err:
          mov       dx,offset toobig wrt dg
std_con_string_output equ Std_Con_String_Output	; NASM port equate
          mov       ah,std_con_string_output
          int       21h
          jmp       command

wrtfilej: jmp   wrtfile
nofilej: jmp    nofile

binload:
          push      ax
          push      dx
          cmp       byte ptr [rdflg],write
          jz        wrtfilej
          call      open_a_file
          jc        nofilej
          mov       bx,[handle]
          mov       ax,(lseek << 8) | 2
          xor       dx,dx
          mov       cx,dx
          int       21h                     ; GET SIZE OF FILE
          mov       si,dx
          mov       di,ax                   ; SIZE TO SI:DI
          mov       ax,(lseek << 8) | 0
          xor       dx,dx
          mov       cx,dx
          int       21h                     ; RESET POINTER BACK TO BEGINNING
          pop       ax
          pop       bx
          push      bx
          push      ax                      ; TRANS ADDR TO BX:AX
          add       ax,15
          mov       cl,4
          shr       ax,cl
          add       bx,ax                   ; Start of transfer rounded up to seg
          mov       dx,si
          mov       ax,di                   ; DX:AX is size
          mov       cx,16
          div       cx
          or        dx,dx
          jz        norem
          inc       ax
norem:                                  ; AX is number of paras in transfer
          add       ax,bx                   ; AX is first seg that need not exist
pdb_block_len equ PDB_block_len	; NASM port label
          cmp       ax,[cs:pdb_block_len]
          ja        no_mem_err
          mov       [cxsave],di
          mov       [bxsave],si
          pop       dx
          pop       ax

rdwr:
; AX:DX is disk transfer address (segment:offset)
; SI:DI is length (32-bit number)

rdwrloop:
          mov       bx,dx                   ; Make a copy of the offset
          and       dx,000fh                ; Establish the offset in 0H-FH range
          nop				; NASM identicalise
          mov       cl,4
          shr       bx,cl                   ; Shift offset and
          add       ax,bx                   ; Add to segment register to get new Seg:offset
          push      ax
          push      dx                      ; Save AX,DX register pair
          mov       word ptr [transadd],dx
          mov       word ptr [transadd+2],ax
          mov       cx,0fff0h               ; Keep request in segment
          or        si,si                   ; Need > 64K?
          jnz       bigrdwr
          mov       cx,di                   ; Limit to amount requested
bigrdwr:
          push      ds
          push      bx
          mov       bx,[handle]
          mov       ah,[rdflg]
          lds       dx,[transadd]
          int       21h                     ; Perform read or write
          pop       bx
          pop       ds
          jc        badwr
          cmp       byte ptr [rdflg],write
          jnz       goodr
          cmp       cx,ax
          jz        goodr
badwr:    mov       cx,ax
          stc
          pop       dx                      ; READ OR WRITE BOMBED OUT
          pop       ax
          ret

goodr:
          mov       cx,ax
          sub       di,cx                   ; Request minus amount transferred
          sbb       si,0                    ; Ripple carry
          or        cx,cx                   ; End-of-file?
          pop       dx                      ; Restore DMA address
          pop       ax
          jz        ret8
          add       dx,cx                   ; Bump DMA address by transfer length
          mov       bx,si
          or        bx,di                   ; Finished with request
          jnz       rdwrloop
ret8:     clc                               ; End-of-file not reached
          ret

nofile:
          mov       dx,offset notfnd wrt dg
restartjmp:
          jmp       restart

wrtfile:
          call      create_a_file           ; Create file we want to write to
          mov       dx,offset noroom wrt dg     ; Creation error - report error
          jc        restartjmp
          mov       si,[bxsave]               ; Get high order number of bytes to transfer
          cmp       si,000fh
          jle       wrtsize                 ; Is bx less than or equal to FH
          xor       si,si                   ; Ignore BX if greater than FH - set to zero
wrtsize:
          mov       dx,offset wrtmes1 wrt dg    ; Print number bytes we are writing
          call      rprbuf
          or        si,si
          jz        nxtbyt
          mov       ax,si
          call      digit
nxtbyt:
          mov       dx,[cxsave]
          mov       di,dx
          call      out16                   ; Amount to write is SI:DI
          mov       dx,offset wrtmes2 wrt dg
          call      rprbuf
          pop       dx
          pop       ax
          call      rdwr
          jnc       closefile
          call      closefile
          call      delete_a_file
          mov       dx,offset nospace wrt dg
          jmp       restartjmp

closefile:
	mov bx, -1
	xchg bx, [handle]
	cmp bx, -1
	je closefile_ret
close equ Close	; NASM port equate
          mov       ah,close
          int       21h
closefile_ret:
          ret

exeload:
          pop       word [retsave]               ; Suck up return addr
          inc       byte ptr [newexec]
          mov       bx,[user_proc_pdb]
          mov       ax,ds
          cmp       ax,bx
          jz        debug_current
          jmp       find_debug

debug_current:
debug_found:
          mov       byte ptr [newexec],0
          push      word [retsave]               ; Get the return address back
          push      cs
          pop       es
          mov       word ptr [com_line+2],cs
          mov       word ptr [com_fcb1+2],cs
          mov       word ptr [com_fcb2+2],cs

          call      exec_a_file
          jc        execerr
          call      set_terminate_vector    ; Reset int 22
          mov       ah,get_current_pdb
          int       21h
          mov       [user_proc_pdb],bx
          mov       [dssave],bx
          mov       [essave],bx
          mov       es,bx
pdb_exit equ PDB_Exit	; NASM port label
          mov       word ptr [es:pdb_exit],offset terminate wrt dg
          mov       word ptr [es:pdb_exit+2],ds
          les       di,[com_csip]
          mov       [cssave],es
          mov       [ipsave],di
          mov       word ptr [disadd+2],es
          mov       word ptr [disadd],di
          mov       word ptr [asmadd+2],es
          mov       word ptr [asmadd],di
          mov       word ptr [defdump+2],es
          mov       word ptr [defdump],di
          mov       bx,ds
          mov       ah,set_current_pdb
          int       21h
          les       di,[com_sssp]
          mov       ax,[es:di]
          inc       di
          inc       di
          mov       [axsave],ax
          mov       [sssave],es
          mov       [spsave],di
          ret

execerr:
          mov       dx,offset notfnd wrt dg
          cmp       ax,exec_file_not_found
          jz        gotexecemes
          mov       dx,offset accmes wrt dg
          cmp       ax,error_access_denied
          jz        gotexecemes
          mov       dx,offset toobig wrt dg
          cmp       ax,exec_not_enough_memory
          jz        gotexecemes
          mov       dx,offset exebad wrt dg
          cmp       ax,exec_bad_format
          jz        gotexecemes
          mov       dx,offset execemes wrt dg
gotexecemes:
          mov       ah,std_con_string_output
          int       21h
          jmp       command

hex2bin:
          mov       [index],dx
          mov       dx,offset hexwrt wrt dg
          cmp       byte ptr [rdflg],write
          jnz       rdhex
          jmp       restartj2
rdhex:
          mov       es,ax
          call      open_a_file
          mov       dx,offset notfnd wrt dg
          jnc       hexfnd
          jmp       restart
hexfnd:
	push bp
	mov bp, sp
	xor ax, ax
	push ax
          mov       si,offset (buffer+bufsiz) wrt dg    ; Flag input buffer as empty
readhex:
          call      getch
          cmp       al,':'                  ; Search for : to start line
          jnz       readhex
          call      getbyt                  ; Get byte count
          mov       cl,al
          mov       ch,0
          jcxz      hexdone
          call      getbyt                  ; Get high byte of load address
          mov       bh,al
          call      getbyt                  ; Get low byte of load address
          mov       bl,al
          add       bx,[index]              ; Add in offset
	jc hex_error
          mov       di,bx
          call      getbyt                  ; Throw away type byte
readln:
          call      getbyt                  ; Get data byte
          stosb
	cmp di, word ptr [bp - 2]	; Check if this is the largest address so far
	jbe havbig
	mov word ptr [bp - 2], di	; Save new largest
havbig:
	test di, di
	jnz nextln
	cmp cx, 1
	ja hex_error
nextln:
          loop      readln
          jmp       short readhex

getch:
          cmp       si,offset (buffer+bufsiz) wrt dg
          jnz       noread
          mov       dx,offset buffer wrt dg
          mov       si,dx
          mov       ah,read
          push      bx
          push      cx
          mov       cx,bufsiz
          mov       bx,[handle]
          int       21h
	cmp cx, ax
	je getch_full
	xchg bx, ax
	mov byte ptr [si + bx], 1Ah
getch_full:
          pop       cx
          pop       bx
noread:
          lodsb
          cmp       al,1ah
          jz        hexdone
          or        al,al
          jnz       ret7
hexdone:
	mov ax, word ptr [bp - 2]
	mov [cxsave], ax
	and word ptr [bxsave], 0
          nop				; NASM identicalise
	mov sp, bp
	pop bp
          ret

hexdig:
          call      getch
          call      hexchk
          jnc       ret7
hex_error:
          mov       dx,offset hexerr wrt dg
restartj2:
          jmp       restart

getbyt:
          call      hexdig
          mov       bl,al
          call      hexdig
          shl       bl,1
          shl       bl,1
          shl       bl,1
          shl       bl,1
          or        al,bl
ret7:     ret


code      ends
          end
